#!/usr/bin/python

#################################################################################
#
# NAME: 	check_snmp_usage.py
#
# COMMENT:  Nagios check to monitor Internet connections that have monthly 
#           usage caps. Requires the creation of a state file. No warning or
#           critical thresholds necessary, will go WARNING if you are headed
#           towards over usage based upon current usage and current date. Will
#           go CRITICAL when you've actually surpassed your allotment. Will use
#           command line SNMP so no extra Python libraries are required.
#
# TODO:     - Everything
#           - As an added feature I'll someday make it work on daily caps and
#           additionally make it so you can adjust the day of the month to reset
#           - Add in the ability to use perfdata as a simple bandwidth
#           method to create graphs in Nagios
#           - SNMP v3?
#           - Still don't have the logic for when the period rolls - I have about
#           29 days to figure this out :)
#
#
# Release Hitory:
#
#################################################################################


import os
import sys
import getopt
import re
import subprocess
import datetime

host = ''
debug = 0
communityString = 'public'
OIDDateTime = '.1.3.6.1.2.1.25.1.2.0'
OIDIfDescr = '.1.3.6.1.2.1.2.2.1.2'
OIDIfInBytes = '.1.3.6.1.2.1.2.2.1.10.'
OIDIfOutBytes = '.1.3.6.1.2.1.2.2.1.16.'

unitMultipliers = {	'b' : 1,
			'B' : 1,
			'k' : 1024,
			'K' : 1024,
			'm' : 1024 * 1024,
			'M' : 1024 * 1024,
			'g' : 1024 * 1024 * 1024,
			'G' : 1024 * 1024 * 1024,
			't' : 1024 * 1024 * 1024 * 1024,
			'T' : 1024 * 1024 * 1024 * 1024,
}

limitMatch = re.compile(r"([0-9]+)([bBkKmMgGtT]{0,1})")

# Find snmpwalk and snmpget

sp = subprocess.Popen(['which', 'snmpget'], stdout=subprocess.PIPE, stderr=subprocess.PIPE)
stdout, stderr = sp.communicate()
if sp.returncode == 0:
	snmpget = stdout.rstrip()
else:
	print('I couldn\'t find snmpget on your system, unable to finish')
	sys.exit(3)

sp = subprocess.Popen(['which', 'snmpwalk'], stdout=subprocess.PIPE, stderr=subprocess.PIPE)
stdout, stderr = sp.communicate()
if sp.returncode == 0:
	snmpwalk = stdout.rstrip()
else:
	print('I couldn\'t find snmpwalk on your system, unable to finish')
	sys.exit(3)


# Functions
def getInterfaces():
	interfaceDict = {}
	walkIfsCmd = [snmpwalk, '-v2c', '-c', communityString, host, OIDIfDescr]
	sp = subprocess.Popen(walkIfsCmd, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
	stdout, stderr = sp.communicate()
	walkOut = stdout.rstrip().split('\n')
	for line in walkOut:
		interfaceDict[line.split()[-1]] = line.split()[0].split('.')[-1]
	return interfaceDict
	

#USAGE
def usage():
        print('Usage: check_snmp_usage.py -H hostname -i interface -b bandwidth limit (bytes) -c community string \n')
try:
        opts, args = getopt.getopt(sys.argv[1:], 'H:i:b:cd')
except getopt.GetoptError as err:
        print(err)
        usage()
        sys.exit(3)

for opt, arg in opts:
        if opt in ('-H'):
                host = arg
        elif opt in ('-i'):
                interface = arg
        elif opt in ('-b'):
                bytesLimit = arg
        elif opt in ('-c'):
                communityString = arg
        elif opt in ('-d'):
                debug = 1
        else:
                usage()
                sys.exit(3)

# Argument processing

if not 'host' in locals():
        print('No host specified')
        sys.exit(3)

if not 'interface' in locals():
	interfaceDict = getInterfaces()
	print('No interface specified - here are your options:')
	for interface in interfaceDict:
		print(interface)
	sys.exit(3)

if not 'bytesLimit' in locals():
	bytesLimit = 0
if not limitMatch.match(bytesLimit):
	print('Could not validate limit argument')
else:
	scalar = limitMatch.search(bytesLimit).group(1)
	multiplier = limitMatch.search(bytesLimit).group(2)
	if multiplier:
		bytesLimit = int(scalar) * int(unitMultipliers[multiplier])



# Not optimized - we'll be getting all of the interfaces from the remote system and match up the indexes
interfaceDict = getInterfaces()

# Now let's grab our counters and the date of the remote system

current = {}
getDateTimeCmd = [snmpget, '-v2c', '-c', communityString, host, OIDDateTime]
sp = subprocess.Popen(getDateTimeCmd, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
stdout, stderr = sp.communicate()
dateTime = stdout.rstrip().split()[-1]
date = dateTime.split(',')[0]
time = dateTime.split(',')[1]
current['year'] = date.split('-')[0]
current['month'] = date.split('-')[1]
current['day'] = date.split('-')[2]
current['hour'] = time.split(':')[0]
current['minute'] = time.split(':')[1]
current['second'] = time.split(':')[2].split('.')[0]

getInBytesCmd = [snmpget, '-v2c', '-c', communityString, host, OIDIfInBytes + interfaceDict[interface]]
getOutBytesCmd = [snmpget, '-v2c', '-c', communityString, host, OIDIfOutBytes + interfaceDict[interface]]

sp = subprocess.Popen(getInBytesCmd, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
stdout, stderr = sp.communicate()
current['inbytes'] = stdout.rstrip().split()[-1]

sp = subprocess.Popen(getOutBytesCmd, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
stdout, stderr = sp.communicate()
current['outbytes'] = stdout.rstrip().split()[-1]

# Time to see if we have a temp file in place or not and read it if we do

tmpFile = '/tmp/check_snmp_usage_' + host + '_' + interface + '.tmp'


if os.path.isfile(tmpFile):
	f = open(tmpFile, 'r')
	fileLines = f.readlines()
	f.close
	lastList = fileLines[-1].rstrip().split('|')
	last = {}
	last['year'] = lastList[0]
	last['month'] = lastList[1]
	last['day'] = lastList[2]
	last['hour'] = lastList[3]
	last['minute'] = lastList[4]
	last['second'] = lastList[5]
	last['inbytes'] = lastList[6]
	last['outbytes'] = lastList[7]
	last['inbytestotal'] = lastList[8]
	last['outbytestotal'] = lastList[9]
else:
	outLine = current['year'] + '|' + current['month'] + '|' + current['day'] + '|' + current['hour'] + '|' + current['minute'] + '|' + current['second'] + '|' + current['inbytes'] + '|' + current['outbytes'] + '|0|0'
	f = open(tmpFile, 'a')
	f.write(outLine)
	f.close()
	print('First run - no status update until the next run')
	sys.exit(3)


# Get all of the differences

format = '%Y %m %d %H %M %S'
lastDateStr = last['year'] + ' ' + last['month'] + ' ' + last['day'] + ' ' + last['hour'] + ' ' + last['minute'] + ' ' + last['second']
currentDateStr = current['year'] + ' ' + current['month'] + ' ' + current['day'] + ' ' + current['hour'] + ' ' + current['minute'] + ' ' + current['second']
lastDateTime = datetime.datetime.strptime(lastDateStr, format)
currentDateTime = datetime.datetime.strptime(currentDateStr, format)

timeDifference = currentDateTime - lastDateTime

if int(current['inbytes']) > int(last['inbytes']) and int(current['outbytes']) > int(last['outbytes']):
	inBytesDifference = int(current['inbytes']) - int(last['inbytes'])
	outBytesDifference = int(current['outbytes']) - int(last['outbytes'])
	inBytesPerSec = int(inBytesDifference) / int(timeDifference.total_seconds())
	outBytesPerSec = int(outBytesDifference) / int(timeDifference.total_seconds())
	inBytesTotal = int(last['inbytestotal']) + int(inBytesDifference)
	outBytesTotal = int(last['outbytestotal']) + int(outBytesDifference)
	outLine = current['year'] + '|' + current['month'] + '|' + current['day'] + '|' + current['hour'] + '|' + current['minute'] + '|' + current['second'] + '|' + current['inbytes'] + '|' + current['outbytes'] + '|' + str(inBytesTotal) + '|' + str(outBytesTotal)
	fileLines[-1] = outLine
	f = open(tmpFile, 'w')
	f.write(outLine)
	f.close()
else:
	print('Seems the counter went whacky, we need to reset things')
	print('expect results after the next check iteration.')
	print('If you continue to see this message something is very wrong.')
	outLine = current['year'] + '|' + current['month'] + '|' + current['day'] + '|' + current['hour'] + '|' + current['minute'] + '|' + current['second'] + '|' + current['inbytes'] + '|' + current['outbytes'] + '|' + last['inbytestotal'] + '|' + last['outbytestotal']
	fileLines[-1] = outLine
	f = open(tmpFile, 'w')
	f.write(outLine)
	f.close()
	sys.exit(3)


# Files should be done, time to do Nagios logic

# Calculate number of seconds into month so far
beginningOfMonthStr = current['year'] + ' ' + current['month'] + ' 1 00 00 00'
beginningOfMonthDateTime = datetime.datetime.strptime(beginningOfMonthStr, format)
secondsIntoMonth = currentDateTime - beginningOfMonthDateTime

# Calculate number of seconds in the current month
if current['month'] == '12':
	nextMonth = '1'
	nextYear = int(current['year']) + 1
else:
	nextMonth = int(current['month']) + 1
	nextYear = current['year']

beginningOfNextMonthStr = str(nextYear) + ' ' + str(nextMonth) + ' 1 00 00 00'
beginningOfNextMonthDateTime = datetime.datetime.strptime(beginningOfNextMonthStr, format)
secondsInMonth = beginningOfNextMonthDateTime - beginningOfMonthDateTime

# Percentage of month complete
monthComplete = ( float(secondsIntoMonth.total_seconds()) / float(secondsInMonth.total_seconds()) ) * 100


# Percentage of data used

# Assume that total allotment is inbytes and outbytes combined

totalDataUsed = int(inBytesTotal) + int(outBytesTotal)
percentageDataUsed = ( float(totalDataUsed) / float(bytesLimit) ) * 100

# Generate perfdata:

perfdata = '|\'BytesInRate\'=' + str(inBytesPerSec) +'bytes/sec;;;;\'BytesOutRate\'=' + str(outBytesPerSec) + 'bytes/sec;;;;'

# Calculate warning/critical

if totalDataUsed > bytesLimit:
	print('CRITICAL - You have exceeded your alloted data usage for this period' + perfdata)
	print('You are alloted ' + str(bytesLimit) + ' bytes for the month but have already used ' + str(totalDataUsed) + ' bytes.')
	sys.exit(2)
elif percentageDataUsed > monthComplete:
	print('WARNING - You are using data at a rate that will cause you to exceed usage allotment for this period' + perfdata)
	print('The month is ' + str(round(monthComplete,2)) + '% complete.')
	print('The data allotment on interface ' + interface + ' is ' + str(round(percentageDataUsed,2)) +'% used.')
	sys.exit(1)
else:
	print('OK - You will not use all of your alloted data for this period at the current rate' + perfdata)
	print('The month is ' + str(round(monthComplete,2)) + '% complete.')
	print('The data allotment on interface ' + interface + ' is ' + str(round(percentageDataUsed,2)) +'% used.')
	sys.exit(0)


if debug == 1:
	 print('The month is ' + str(round(monthComplete,2)) + '% complete.')
	 print('The data allotment on interface ' + interface + ' is ' + str(round(percentageDataUsed,2)) +'% used.')
	 print('The current incoming rate is ' + str(inBytesPerSec) + ' bytes/second.')
	 print('The current outgoing rate is ' + str(outBytesPerSec) + ' bytes/second.')
