#!/usr/bin/python

#
# Nagios check to identify age of all snapshots on an ESX host (not vCenter)
# 
# Will take warning and critical arguments from command line
#
# Authentication should be in a 600 file somewhere accessible by nagios user
#
# Authentication file should just have the password, nothing else.
#
# User to be specified at the command line

import sys
import os
import getopt
import re
import paramiko
import datetime

username = "root"
password = 0
passfile = 0
warning = 0
critical = 0
warnList = []
critList = []
debug = 0


ssh = paramiko.SSHClient()
ssh.set_missing_host_key_policy(paramiko.AutoAddPolicy())


# USAGE
def usage():
	print "usage: check_snapshot.py -H hostname [-U username] <-P password | -f PasswordFile>\n"
	print "PasswordFile should be a document readable by the nagios user only containing the" 
	print "password for the specified user. If no user is specified the check will be run as root."
	print "\nUse the -w and -c arguments to indicate the number of days you wish to use for warning and critical levels. If you do not specify these arguments the return will be \"OK\" and there will just be debug output"

# Get Command Line Arguments
try:
	opts, args = getopt.getopt(sys.argv[1:], "H:U:P:f:c:w:", ["hostname=", "username=", "password="])
except getopt.GetoptError as err:
	print err
	usage()
	sys.exit(3)

for opt, arg in opts:
	if opt in ('-H', '--hostname'):
		hostname = arg
	elif opt in ('-U', '--username'):
		username = arg
	elif opt in ('-P', '--password'):
		password = arg
	elif opt in ('-f'):
		passfile = arg
	elif opt in ('-c'):
		critical = datetime.timedelta(days = int(arg))
	elif opt in ('-w'):
		warning = datetime.timedelta(days = int(arg))
	else:
		usage()
		sys.exit(3)

# Command Line Argument tests
if not password and not passfile:
	print "No password specified"
	usage()
	sys.exit(3)

if password and passfile:
	print "Cannot specify both password file and command line password"
	usage()
	sys.exit(3)

if passfile and not os.path.isfile(passfile):
	print "Cannot find specified password file"
	usage()
	sys.exit(3)

if passfile:
	lines = 0
	with open(passfile) as file:
		for line in file:
			if line.strip():
				lines += 1
	if not lines:
		print 'Password file specified is empty'
		sys.exit(3)
	elif lines > 1:
		print 'More than 1 line in password file'
		usage()
		sys.exit(3)
	else:
		with open(passfile) as file:
			password = file.readline().rstrip()

if not hostname:
	print "No hostname specified"
	usage()
	sys.exit(3)

if not warning and not critical:
	debug = 1

# Snapshot class definition

class snapshot:
	def __init__(self, vmsnFullPath):
		self.vmsnFullPath = vmsnFullPath
	vmFolder = ""
	vmsnFile = ""
	vmxFile = ""
	displayName = ""

	def getAge(self):
		ssh.connect(hostname, username=username, password=password)
		path = self.vmsnFullPath
		stdin, stdout, stderr = ssh.exec_command('stat -c %y ' + path.replace(" ", "\\ "))
		type(stdin)
		ageList = stdout.readlines()
		for age in ageList:
			date = re.sub('\..*$', '', age).rstrip()
		ssh.close()
 		self.date = datetime.datetime.strptime(date, '%Y-%m-%d %H:%M:%S')
		self.age = currentDate - self.date


# Connect to ESX host and get the initial list of VMs with Snapshots

findSnaps = str("find /vmfs/volumes -iname '*vmsn'")
ssh.connect(hostname, username=username, password=password)
stdin, stdout, stderr = ssh.exec_command('find /vmfs/volumes -iname \'*vmsn\'')
type(stdin)
vmsnFullPath = stdout.readlines()

# Get current datetime from ESX host - don't want to or need to know TZinfo

stdin, stdout, stderr = ssh.exec_command('date +%F\ %T')
type(stdin)
currentDate = stdout.readlines()
for date in currentDate:
	currentDate = date.rstrip()
currentDate = datetime.datetime.strptime(currentDate, '%Y-%m-%d %H:%M:%S')

# Create all of our snapshot instances

snapshots = []
for snapPath in vmsnFullPath:
	snapshots.append(snapshot(snapPath.rstrip()))

# Create a list of VM folders with snapshotted VMs inside

pattern = '^.*/'
for snapshot in snapshots:
	string = snapshot.vmsnFullPath
	match = re.search(pattern, string)
	snapshot.vmFolder = match.group(0).rstrip()


pattern = '[^/]+?$'
vmsnFile = []
for snapshot in snapshots:
	string = snapshot.vmsnFullPath
	match = re.search(pattern, string)
	snapshot.vmsnFile = match.group(0).rstrip()

# Get the VMX files

for snapshot in snapshots:
	path = snapshot.vmFolder
	stdin, stdout, stderr = ssh.exec_command('find ' + path.replace(" ", "\\ ") + ' -iname \'*vmx\'')
	type(stdin)
	output = stdout.readlines()
	for line in output:
		snapshot.vmxFile = line.rstrip()

for snapshot in snapshots:
	vmxFile = snapshot.vmxFile
	stdin, stdout, stderr = ssh.exec_command('grep "^displayName" ' + vmxFile.replace(" ", "\\ "))
	type(stdin)
	output = stdout.readlines()
	for line in output:
		match = re.search('"([^"]*)"', line)
		snapshot.displayName = match.group(0).replace('"', '')

ssh.close()

for snapshot in snapshots:
	snapshot.getAge()

	if critical:
		if snapshot.age > critical:
			critList.append(snapshot)
	if warning:
		if snapshot.age > warning and snapshot.age < critical:
			warnList.append(snapshot)

# Nagios exit logic

if len(critList) > 0:
	exitCode = 2
elif len(warnList) > 0:
	exitCode = 1
else:
	exitCode = 0

exitStatus = ""
if warnList or critList:
	if warnList:
		exitStatus = exitStatus + str(len(warnList)) + ' VMs are WARNING\n'
	if critList:
		exitStatus = exitStatus + str(len(critList)) + ' VMs are CRITICAL\n'
for snapshot in warnList:
	prettyAge = re.sub(',.*$', '', str(snapshot.age))
	exitStatus = exitStatus + 'Guest ' + snapshot.displayName + ' has snapshot ' + prettyAge + ' old!\n'

for snapshot in critList:
	prettyAge = re.sub(',.*$', '', str(snapshot.age))
	exitStatus = exitStatus + 'Guest ' + snapshot.displayName + ' has snapshot ' + prettyAge + ' old!\n'

if debug:
	exitStatus = exitStatus + str(len(snapshots)) + ' snapshots exist.\n'
	for snapshot in snapshots:
		prettyAge = re.sub(',.*$', '', str(snapshot.age))
		exitStatus = exitStatus + 'Guest ' + snapshot.displayName + ' has snapshot ' + prettyAge + ' old!\n'


print exitStatus
sys.exit(exitCode)